/*
 *
 *   BaseZ
 *
 *   Copyright (C) 2013  Milan Kupcevic
 *
 *   You can redistribute and/or modify this software under the
 *   terms of the GNU General Public License version 3, or any later
 *   version as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   GPLv3+
 *
 *
 *   Encodings per RFC 4648:
 *    base64     Base 64 Encoding
 *    base64url  Base 64 Encoding with URL and Filename Safe Alphabet
 *    base32     Base 32 Encoding
 *    base32hex  Base 32 Encoding with Extended Hex Alphabet
 *    base16     Base 16 Encoding
 *
 *   Encoding per RFC 2045:
 *    base64mime MIME base64 Content-Transfer-Encoding
 *
 *   Encoding per RFC 1421:
 *    base64pem  PEM Printable Encoding
 *
 */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "basez.h"

typedef enum opt_base {b16, b32, b32hex, b64, b64url, b64mime, b64pem} opt_base;
typedef enum opt_mode {encode, decode} opt_mode;
typedef enum opt_crlf {off, on} opt_crlf;
typedef enum code {regular, hex, url, mime, pem} code;

const char* base_name(const char *command_path_name);
void version();
void help();
void encode_b64(const code table);
void decode_b64(const code table);
void encode_b32(const code table);
void decode_b32(const code table);
void encode_b16();
void decode_b16();
void invalid_input();
void internal_error();
void sys_err(const char *msg);
void newline_out();

static const char *command_name;
static opt_crlf o_crlf = off;
static FILE *file_in;

int
main(
  const int argc,
  const char *argv[])
{
  int i;
  const char *filename_in = NULL;
  opt_base o_b = b64;
  opt_mode o_m = encode;
  command_name = NULL;

  if (argc > 0)
    command_name = base_name(argv[0]);

  if (command_name == NULL)
    command_name = "basez";

  file_in = stdin;

  if (strcmp(command_name, "base16") == 0)     o_b = b16;
  if (strcmp(command_name, "base32") == 0)     o_b = b32;
  if (strcmp(command_name, "base32hex") == 0)  o_b = b32hex;
  if (strcmp(command_name, "base64plain") == 0)     o_b = b64;
  if (strcmp(command_name, "base64url") == 0)  o_b = b64url;
  if (strcmp(command_name, "base64mime") == 0) o_b = b64mime;
  if (strcmp(command_name, "base64pem") == 0)  o_b = b64pem;

  for(i = argc - 1; i > 0; i--)
  {
    if (strcmp(argv[i], "--version") == 0)   { version(); }
    if (strcmp(argv[i], "--help") == 0)      { help(); }
    if (strcmp(argv[i], "--b16") == 0)       { o_b = b16; continue; }
    if (strcmp(argv[i], "--b32") == 0)       { o_b = b32; continue; }
    if (strcmp(argv[i], "--b32hex") == 0)    { o_b = b32hex; continue; }
    if (strcmp(argv[i], "--b64") == 0)       { o_b = b64; continue; }
    if (strcmp(argv[i], "--b64url") == 0)    { o_b = b64url; continue; }
    if (strcmp(argv[i], "--b64mime") == 0)   { o_b = b64mime; continue; }
    if (strcmp(argv[i], "--b64pem") == 0)    { o_b = b64pem; continue; }
    if (strcmp(argv[i], "--crlf") == 0)      { o_crlf = on; continue; }
    if((strcmp(argv[i], "-d") == 0) || 
       (strcmp(argv[i], "--decode") == 0))   { o_m = decode; continue; }

    if (*argv[i] == '-')
    { 
      fprintf(stderr, "%s: %s: Invalid option\n", command_name, argv[i]);
      exit(EXIT_FAILURE);
    }

    if (filename_in == NULL)
    {
      filename_in = argv[i];
    }
    else
    {
      fprintf(stderr, "%s: Only one input file is allowed\n", command_name);
      exit(EXIT_FAILURE);
    }
  }

  if (filename_in != NULL)
  {
    file_in = fopen(filename_in, o_crlf == on && o_m == encode ? "rt" : "rb");
    if (file_in == NULL)
      sys_err(filename_in);
  }

  switch(o_m)
  { 
    case encode:
      switch(o_b)
      {
        case b16:
          encode_b16();
          break;
        case b32:
          encode_b32(regular);
          break;
        case b32hex:
          encode_b32(hex);
          break;
        case b64:
          encode_b64(regular);
          break;
        case b64url:
          encode_b64(url);
          break;
        case b64mime:
          encode_b64(mime);
          break;
        case b64pem:
          encode_b64(pem);
          break;
        default:
          internal_error();
      }
      break;
    case decode:
      switch(o_b)
      {
        case b16:
          decode_b16();
          break;
        case b32:
          decode_b32(regular);
          break;
        case b32hex:
          decode_b32(hex);
          break;
        case b64:
          decode_b64(regular);
          break;
        case b64url:
          decode_b64(url);
          break;
        case b64mime:
          decode_b64(mime);
          break;
        case b64pem:
          decode_b64(pem);
          break;
        default:
          internal_error();
      }
      break;
    default:
      internal_error();
  }

  return EXIT_SUCCESS;
}

const char*
base_name(
  const char *command_path_name)
{
  if (command_path_name == NULL)
    return NULL;

  const char *token;
  const char *ret = NULL;

  token = command_path_name;

  while (*token != '\0')
  {
    ret = token;
    token = strpbrk(token, "\\/");
    if(token == NULL) break;
    token++;
  }
  return ret;
}

void
version ()
{
  printf("%s (", command_name);
  puts(
#if HAS_ORG
  ORG " "
#endif
APPNAME ") " VERSION "\n"
"Copyright (C) " COPYRIGHT_YEARS "  " COPYRIGHT_NAMES "."
#if TMCLAIM
  "\n" TRADEMARKS "."
#endif
  );
  puts(
"This program comes with ABSOLUTELY NO WARRANTY; not even for MERCHANTABILITY\n"
"or FITNESS FOR A PARTICULAR PURPOSE. This program is licensed under the\n"
"terms of the GNU GPL version 3 or any later version as published by the Free\n"
"Software Foundation. User documentation is licensed under the Creative\n"
"Commons Attribution-ShareAlike 3.0 license.\n"
"\n"
"Written by " AUTHORS "."
  );

  exit(EXIT_SUCCESS);
}

void
help ()
{
  puts(
"\n"
"Usage: basez  [OPTION]... [FILE]\n"
"   or: base16 [OPTION]... [FILE]\n"
"   or: base32 [OPTION]... [FILE]\n"
"   or: base32hex [OPTION]... [FILE]\n"
"   or: base64plain [OPTION]... [FILE]\n"
"   or: base64url [OPTION]... [FILE]\n"
"   or: base64mime [OPTION]... [FILE]\n"
"   or: base64pem [OPTION]... [FILE]\n"
"\n"
#if HAS_ORG
  ORG " " 
#endif
APPNAME 
" encodes/decodes base16, base32, base32hex, base64 or base64url data \n"
"stream per RFC 4648; MIME base64 Content-Transfer-Encoding per RFC 2045; \n"
"or PEM Printable Encoding per RFC 1421.\n");
  puts(
"Base16 encoding produces a multiple of two-character blocks in hexadecimal \n"
"notation [0-9A-F]. It needs no padding and preserves the sort order of the \n"
"encoded data. Decoding is case insensitive.\n"
  );
  puts(
"Base32 encoded stream is a multiple of eight-character blocks consisting of \n"
"letters and numbers [A-Z2-7]. Numbers easily confused with some letters \n"
"are skipped intentionally to make this encoding suitable for storage on, or \n"
"transport over any medium or data transport mechanism, including "
  );
  puts(
"non-case-preserving barcodes or printed out strings that could be spelled \n"
"out and typed in by humans.  If needed, the last encoded block is padded \n"
"with equal sign end padding. Appearance of the padding character [=] at the \n"
"end of the encoded steam can be avoided by encoding data of size divisible \n"
"by 5. Base32 decoding is case insensitive.\n"
  );
  puts(
"Base32hex encoding works the same way as base32, but with extended hex\n"
"character set [0-9A-V] to  preserve the sort order of the encoded data. \n"
"This encoding should not be confused with base32.\n"
  );
  puts(
"Base64 encoded stream is a multiple of four-character blocks using \n"
"uppercase letters, lowercase letters, numbers, plus and slash \n"
"[A-Za-z0-9+/]. It uses equal sign [=] for end padding. Base64 decoding \n"
"is case sensitive. It has an option to convert local native text line \n"
"breaks into canonical CRLF sequences prior to encoding; or to convert CRLF \n"
"sequences into native text line breaks after the decoding.\n"
  );
  puts(
"Base64url encoding is technically the same as base64, but instead of the \n"
"plus and slash signs [+/], it uses minus and underscore [-_]. Appearance of \n"
"the padding character [=] in the encoded stream can be avoided by encoding \n"
"data of size divisible by 3.\n"
  );
  puts(
"Base64mime and base64pem are the same encodes as base64, but with encoded \n"
"stream line length limit of 76 and 64 characters respectively. MIME and \n"
"PEM decoding ignore all white and non-printable characters in the encoded \n"
"stream, except backspace.\n"
  );
  puts(
"\n"
"Options:\n"
"    -d, --decode     Decode.\n"
"\n"
"        --b16        base16 coding -- default for base16 command.\n"
"\n"
"        --b32        base32 coding -- default for base32 command.\n"
"        --b32hex     base32hex coding -- default for base32hex command.\n"
"\n"
"        --b64        base64 coding -- default for basez, base64plain command.\n"
"        --b64url     base64url coding -- default for base64url command.\n"
"        --b64mime    MIME base64 coding -- default for base64mime command.\n"
  );
  puts(
"        --b64pem     PEM printable coding -- default for base64pem command.\n"
"        --crlf       Convert native text line breaks into CRLF sequences \n"
"                     prior to base64 encoding; or convert CRLF sequences \n"
"                     into native text line breaks after base64 decoding.\n"
"\n"
"        --help       Display help.\n"
"        --version    Display program version information.\n"
"\n"
"When no FILE is specified, read standard input.\n"
"\n"
  );
  puts(
"Examples:\n"
"Base32 encode a string:\n"
"\n"
"    printf \"Hello\" | base32 \n"
"\n"
"Base64 encode a file per RFC 4648\n"
"\n"
"    base64plain file > encoded.base64.file\n"
"\n"
"Decode it back:\n"
"\n"
"    base64plain -d encoded.base64.file > file\n"
"\n"
"MIME base64 encode a file per RFC 2045:\n"
"\n"
"    base64mime file > encoded.MIMEbase64.file\n"
"\n"
  );
  puts(
"Report bugs to <bug-" EMAIL "> with a 'bug:' somewhere in the subject line.\n"
"\n"
"These commands are parts of the "
#if HAS_ORG
  ORG " " 
#endif
APPNAME " "
"software package.\n"
"\n"
"    [" 
#if HAS_ORG
  ORG " "
#endif
APPNAME "]        <http://" WEB ">\n"
  );

  exit(EXIT_SUCCESS);
}

void
encode_b64(
  const code table)
{
  char *buffin_tmp;
  char buff_tmp[4605]; /* (buffin / 2 - 3) */
  char buffout[12480]; /* (4 x 3072 + 4 x 3072 / 64) */
  char buffin[9216]; /* (3 x 3072) */
  size_t bs;
  size_t blocks;
  size_t rd;
  size_t *bbytes_tmp;
  size_t bbytes = 0;
  size_t bby_tmp = 0;
  size_t buffer_out_count = 0;
  size_t buffer_out_max = 0;
  size_t nl;
  size_t full_units;

  if (o_crlf == on)
  {
    bs = 4605;
    buffin_tmp = buff_tmp;
    bbytes_tmp = &bby_tmp;
  }
  else
  {
    bs = 9216;
    buffin_tmp = buffin;
    bbytes_tmp = &bbytes;
  }

  if (table == pem)
    buffer_out_max = 16;
  else if (table == mime)
    buffer_out_max = 19;

  do
  {
    if (bbytes < bs)
    {
      do
      {
        rd = fread(&buffin_tmp[*bbytes_tmp], 1, bs - *bbytes_tmp, file_in);
        *bbytes_tmp += rd;
      } while (*bbytes_tmp < bs && !ferror(file_in) && !feof(file_in));

      if (ferror(file_in))
        sys_err("");
    }

    if (o_crlf == on)
    {
      size_t i;
      for (i = 0; i < *bbytes_tmp;  i++)
      {
        switch(buffin_tmp[i])
        {
          case '\n':
            buffin[(bbytes)++] = '\r';
            buffin[(bbytes)++] = '\n';
            break;
          default:
            buffin[(bbytes)++] = buffin_tmp[i];
        }
      }
      *bbytes_tmp = 0;
    }

    nl = 0;
    full_units = bbytes / 3;

    for (blocks = 0; blocks < full_units; blocks++)
    {
      switch(table)
      {
        case pem: 
        case mime: 
          if (buffer_out_count >= buffer_out_max)
          {
            buffer_out_count = 0;
            buffout[blocks * 4 + nl] = '\n';
            nl++;
          }
          buffer_out_count++;
        case regular: 
          encode_base64(&buffin[blocks * 3], 3, &buffout[blocks * 4 + nl]);
          break;
        case url: 
          encode_base64url(&buffin[blocks * 3], 3, &buffout[blocks * 4]);
          break;
        default:
          internal_error();
      }
    }

    if (bbytes % 3 != 0)
    {
      if (feof(file_in))
      {
        switch(table)
        {
          case pem: 
          case mime: 
            if (buffer_out_count >= buffer_out_max)
            {
              buffer_out_count = 0;
              buffout[blocks * 4 + nl] = '\n';
              nl++;
            }
            buffer_out_count++;
          case regular: 
            encode_base64(&buffin[blocks*3],bbytes%3, &buffout[blocks*4+nl]);
            break;
          case url: 
            encode_base64url(&buffin[blocks*3],bbytes%3, &buffout[blocks*4]);
            break;
          default:
            internal_error();
        }
        blocks++;
        bbytes = 0;
      }
      else
      {
        bbytes = bbytes % 3;
        memmove(buffin, &buffin[blocks*3], bbytes);
      }
    }
    else
      bbytes = 0;

    fwrite(buffout, 1, blocks * 4 + nl, stdout);
    if (ferror(stdout))
      sys_err("");

  } while (!feof(file_in) || bbytes > 0);

  if ((table == mime) || (table == pem))
    newline_out();
}

void
decode_b64(
  const code table)
{
  char *buffin_tmp;
  char buff_tmp[8192]; /* 4 x 2048 */
  char buffin[8192]; /* 4 x 2048 */
  char buffout[6144]; /* 3 x 2048 */
  size_t bbytes = 0;
  size_t bby_tmp = 0;
  size_t *bbytes_tmp;
  size_t bout_bytes;
  size_t blocks;
  size_t i;
  size_t rd;
  size_t ret_bytes;
  size_t full_units;
  int end_of_coding = 0;

  if ((table == mime) || (table == pem))
  {
    buffin_tmp = buff_tmp;
    bbytes_tmp = &bby_tmp;
  }
  else
  {
    buffin_tmp = buffin;
    bbytes_tmp = &bbytes;
  }

  do
  {
    do
    {
      rd = fread(&buffin_tmp[*bbytes_tmp], 1, 8192 - *bbytes_tmp, file_in);
      *bbytes_tmp += rd;
    } while (*bbytes_tmp < 8192 && !ferror(file_in) && !feof(file_in));

    if (ferror(file_in))
      sys_err("");

    if ((table == mime) || (table == pem))
    {
      for (i = 0; i < *bbytes_tmp; i++)
      {
        if (buffin_tmp[i] > 0x20)
          buffin[bbytes++] = buffin_tmp[i];
      }
      *bbytes_tmp = 0;
    }

    bout_bytes = 0;
    full_units = bbytes / 4;

    for (blocks = 0; blocks < full_units; blocks++)
    {
      if (end_of_coding != 0)
        invalid_input();

      switch(table)
      {
        case pem: 
        case mime: 
        case regular: 
          ret_bytes = decode_base64(&buffin[blocks*4], &buffout[bout_bytes]);
          break;
        case url: 
          ret_bytes = decode_base64url(&buffin[blocks*4], &buffout[bout_bytes]);
          break;
        default:
          internal_error();
      }

      switch(ret_bytes)
      {
        case 0:
          invalid_input();
        case 1:
        case 2:
          end_of_coding = 1;
        case 3:
          if (o_crlf == on)
          { /* remove CR characters from the decoded stream */
            for(i = 0; i < ret_bytes; i++)
            {
              while (buffout[bout_bytes + i] == '\r' && i < ret_bytes)
              {
                if (i + 1 == ret_bytes)
                  ret_bytes--;
                else
                  memmove(&buffout[bout_bytes + i],
                          &buffout[bout_bytes + i + 1],
                          --ret_bytes - i);
              }
            }
          }
          bout_bytes += ret_bytes;
          break;
        default:
          internal_error();
      }
    }

    if (bbytes % 4 != 0)
    {
      if (feof(file_in))
      {
        invalid_input();
      }
      else
      {
        bbytes = bbytes % 4;
        memmove(buffin, &buffin[blocks*4], bbytes);
      }
    }
    else
      bbytes = 0;

    if (bout_bytes > 0)
    {
      fwrite(buffout, 1, bout_bytes, stdout);
      if (ferror(stdout))
        sys_err("");
    }
  } while (!feof(file_in));
}

void
encode_b32(
  const code table)
{
  char buffin[5120]; /* 5 x 1024 */
  char buffout[8192]; /* 8 x 1024 */
  size_t bbytes;
  size_t blocks;
  size_t rd;
  unsigned int full_units;

  do
  {
    bbytes = 0;
    do
    {
      rd = fread(&buffin[bbytes], 1, 5120 - bbytes, file_in);
      bbytes += rd;
    } while (bbytes < 5120 && !ferror(file_in) && !feof(file_in));

    if (ferror(file_in))
      sys_err("");

    full_units = bbytes / 5;

    for (blocks = 0; blocks < full_units; blocks++)
    {
      switch(table)
      {
        case regular: 
          encode_base32(&buffin[blocks * 5], 5, &buffout[blocks * 8]);
          break;
        case hex:
          encode_base32hex(&buffin[blocks * 5], 5, &buffout[blocks * 8]);
          break;
        default:
          internal_error();
       }
    }

    if (bbytes % 5 != 0)
    {
      switch(table)
      {
        case regular: 
          encode_base32(&buffin[blocks * 5], bbytes % 5, &buffout[blocks * 8]);
          break;
        case hex:
          encode_base32hex(&buffin[blocks*5], bbytes % 5, &buffout[blocks * 8]);
          break;
        default:
          internal_error();
       }
       blocks++;
    }

    if (bbytes > 0)
    {
      fwrite(buffout, 1, blocks * 8, stdout);
      if (ferror(stdout))
        sys_err("");
    }
  } while (!feof(file_in));
}

void
decode_b32(
  const code table)
{
  char buffin[8192]; /* 8 x 1024 */
  char buffout[5120]; /* 5 x 1024 */
  size_t bbytes;
  size_t blocks;
  size_t rd;
  unsigned int full_units;
  size_t ret_bytes = 0;
  size_t bout_bytes;
  int end_of_coding = 0;

  do
  {
    bbytes = 0;
    do
    {
      rd = fread(&buffin[bbytes], 1, 8192 - bbytes, file_in);
      bbytes += rd;
    } while (bbytes < 8192 && !ferror(file_in) && !feof(file_in));

    if (ferror(file_in))
      sys_err("");

    if (bbytes % 8 != 0)
      invalid_input();

    bout_bytes = 0;
    full_units = bbytes / 8;

    for (blocks = 0; blocks < full_units; blocks++)
    {
      if (end_of_coding != 0)
        invalid_input();

      switch(table)
      {
        case regular: 
          ret_bytes = decode_base32(&buffin[blocks*8], &buffout[blocks*5]);
          break;
        case hex: 
          ret_bytes = decode_base32hex(&buffin[blocks*8], &buffout[blocks*5]);
          break;
        default:
          internal_error();
      }
      switch(ret_bytes)
      {
        case 0:
          invalid_input();
        case 1:
        case 2:
        case 3:
        case 4:
          end_of_coding = 1;
        case 5:
          bout_bytes += ret_bytes;
          break;
        default:
          internal_error();
      }
    }

    if (bout_bytes > 0)
    {
      fwrite(buffout, 1, bout_bytes, stdout);
      if (ferror(stdout))
        sys_err("");
    }
  } while (!feof(file_in));
}

void 
encode_b16()
{
  char buffin[4096]; /* 4 x 1024 */
  char buffout[8192]; /* 8 x 1024 */
  size_t bbytes;
  size_t count;
  size_t rd;

  do
  {
    bbytes = 0;
    do
    {
      rd = fread(&buffin[bbytes], 1, 4096 - bbytes, file_in);
      bbytes += rd;
    } while (bbytes < 4096 && !ferror(file_in) && !feof(file_in));

    if (ferror(file_in))
      sys_err("");

    for (count = 0; count < bbytes; count++)
    {
      encode_base16(buffin[count], &buffout[count * 2]);
    }

    if (count > 0)
    {
      fwrite(buffout, 1, count * 2, stdout);
      if (ferror(stdout))
        sys_err("");
    }

  } while (!feof(file_in));
}

void 
decode_b16()
{
  char buffin[8192]; /* 8 x 1024 */
  char buffout[4096]; /* 4 x 1024 */
  size_t bbytes;
  size_t count;
  size_t rd;

  do
  {
    bbytes = 0;
    do
    {
      rd = fread(&buffin[bbytes], 1, 8192 - bbytes, file_in);
      bbytes += rd;
    } while (bbytes < 8192 && !ferror(file_in) && !feof(file_in));

    if (ferror(file_in))
      sys_err("");

    if (bbytes % 2 != 0)
      invalid_input();

    for (count = 0; count < bbytes; count += 2)
    {
      if (decode_base16(&buffin[count], &buffout[count / 2]) == 0)
        invalid_input();
    }

    if (count > 0)
    {
      fwrite(buffout, 1, count / 2, stdout);
      if (ferror(stdout))
        sys_err("");
    }

  } while (!feof(file_in));
}

void
internal_error()
{
  fprintf(stderr, "%s: Internal error\n", command_name);
  exit(EXIT_FAILURE);
}

void
invalid_input()
{
  fprintf(stderr, "%s: Invalid input\n", command_name);
  exit(EXIT_FAILURE);
}

void
sys_err(
  const char *msg)
{
  if (msg != NULL)
  {
    if (*msg != '\0')
    {
      fprintf(stderr, "%s: %s: %s\n", command_name, msg, strerror(errno));
      exit(EXIT_FAILURE);
    }
  }

  perror(command_name);
  exit(EXIT_FAILURE);
}

void
newline_out()
{
  fwrite("\n", 1, 1, stdout);
  if (ferror(stdout))
    sys_err("");
}

